"use client";

import React, { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Alert, AlertDescription, AlertTitle } from "../components/ui/alert";
import { Button } from "../components/ui/button";
import { AlertCircle, BellOff } from "lucide-react";
import { useAuth } from "../hooks/use-auth";
import { format } from 'date-fns';
import { useTranslation } from 'react-i18next';

interface SystemNotification {
  device_id: string;
  thresholdKey: string;
  message: string;
  timestamp: string;
}

const SystemNotificationsTab: React.FC = () => {
  const { user } = useAuth();
  const { t, i18n } = useTranslation();
  const [notifications, setNotifications] = useState<SystemNotification[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Проверка перевода
  console.log('SystemNotificationsTab - Current language:', i18n.language);
  console.log('SystemNotificationsTab - Test translation:', t('notifications.message80', 'Device has reached 80% of max value'));

  useEffect(() => {
    const fetchNotifications = async () => {
      if (!user?.email) {
        setNotifications([]);
        setLoading(false);
        return;
      }
      setLoading(true);
      setError(null);
      try {
        // Fetch notifications for all devices
        const response = await fetch('/api/postgres/device-threshold-flags', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ email: user.email, allDevices: true }) // Added allDevices flag
        });
        if (!response.ok) {
          const errorData = await response.json();
          throw new Error(errorData.error || t('errors.fetchFailed', 'Failed to fetch notifications'));
        }
        const data = await response.json();
        setNotifications(data.notifications || []);
      } catch (err: any) {
        setError(err.message || t('errors.fetchFailed', 'Failed to fetch notifications'));
        setNotifications([]);
      } finally {
        setLoading(false);
      }
    };
    fetchNotifications();
  }, [user, t]);

  // Функция для перевода ключей порогов
  const getThresholdTranslation = (thresholdKey: string) => {
    const thresholdMap: Record<string, string> = {
      '80%': t('notifications.threshold80', '80% threshold reached'),
      '85%': t('notifications.threshold85', '85% threshold reached'),
      '90%': t('notifications.threshold90', '90% threshold reached'),
      '100%': t('notifications.threshold100', '100% threshold reached'),
    };
    return thresholdMap[thresholdKey] || thresholdKey;
  };

  // Функция для перевода сообщений
  const getMessageTranslation = (message: string) => {
    console.log('SystemNotificationsTab - Original message:', message);
    
    // Динамический маппинг для любого устройства
    const deviceMatch = message.match(/Device (.+) reached (80%|85%|90%|100%) usage threshold\./);
    if (deviceMatch) {
      const [, deviceName, threshold] = deviceMatch;
      const thresholdKey = threshold as '80%' | '85%' | '90%' | '100%';
      const translationKey = `notifications.message${thresholdKey.replace('%', '')}`;
      console.log('SystemNotificationsTab - Translation key:', translationKey);
      const translated = t(translationKey, `Device ${deviceName} has reached ${threshold} of max value`);
      console.log('SystemNotificationsTab - Translated message:', translated);
      return translated;
    }
    
    console.log('SystemNotificationsTab - No match found, returning original');
    return message;
  };

  const handleClearNotifications = () => {
    setNotifications([]);
  };

  return (
    <Card className="w-[90vw] mx-auto">
      <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
        <CardTitle className="text-sm font-medium">{t('systemNotificationsTab.title', 'System Notifications')}</CardTitle>
        {notifications.length > 0 && (
          <Button 
            variant="outline" 
            size="sm" 
            onClick={handleClearNotifications}
            className="bg-[#D74322] text-white border-[#D74322] hover:bg-[#D74322]/90 hover:text-white"
          >
            <BellOff className="mr-2 h-4 w-4" /> {t('systemNotificationsTab.clearAll', 'Clear All')}
          </Button>
        )}
      </CardHeader>
      <CardContent>
        {loading ? (
          <p className="text-sm text-muted-foreground text-center py-4">{t('systemNotificationsTab.loading', 'Loading notifications...')}</p>
        ) : error ? (
          <Alert variant="destructive">
            <AlertCircle className="h-4 w-4" />
            <AlertTitle>{t('systemNotificationsTab.error', 'Error')}</AlertTitle>
            <AlertDescription>{error}</AlertDescription>
          </Alert>
        ) : notifications.length === 0 ? (
          <div className="flex flex-col items-center justify-center py-10 text-muted-foreground">
            <BellOff className="h-12 w-12 mb-4" />
            <p className="text-lg">{t('systemNotificationsTab.noNotifications', 'No system notifications at the moment.')}</p>
            <p className="text-sm">{t('systemNotificationsTab.thresholdAlerts', 'Threshold alerts from devices will appear here.')}</p>
          </div>
        ) : (
          <div className="space-y-4 max-h-[calc(100vh-20rem)] overflow-y-auto pr-2">
            {notifications
              .slice()
              .sort((a, b) => new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime())
              .map((notification, idx) => (
                <Alert 
                  key={notification.device_id + notification.thresholdKey + idx}
                  variant={notification.thresholdKey === '100%' ? "destructive" : "default"}
                  className="shadow-sm"
                >
                  <AlertCircle className="h-4 w-4" />
                  <span className="text-xs text-muted-foreground whitespace-nowrap pl-2">
                      {notification.timestamp ? format(new Date(notification.timestamp), 'PPpp') : ''}
                    </span>
                  <AlertTitle className="flex justify-between items-start text-sm">
                  
                    <span className="font-semibold">
                      {t('systemNotificationsTab.device', 'Device')}: {notification.device_id}
                      <span className="font-normal ml-1"> ({getThresholdTranslation(notification.thresholdKey)})</span>
                    </span>
                    
                  </AlertTitle>
                  <AlertDescription className="text-xs leading-relaxed">
                    {getMessageTranslation(notification.message)}
                  </AlertDescription>
                </Alert>
              ))}
          </div>
        )}
      </CardContent>
    </Card>
  );
};

export default SystemNotificationsTab; 