import { NextResponse } from 'next/server';
import admin from 'firebase-admin';
import { postgresClient } from '@/lib/postgresql-client';

// Initialize Firebase Admin SDK if not already initialized
if (!admin.apps.length) {
  const serviceAccount = {
    type: process.env.FIREBASE_TYPE,
    project_id: process.env.FIREBASE_PROJECT_ID,
    private_key_id: process.env.FIREBASE_PRIVATE_KEY_ID,
    private_key: process.env.FIREBASE_PRIVATE_KEY?.replace(/\\n/g, '\n'),
    client_email: process.env.FIREBASE_CLIENT_EMAIL,
    client_id: process.env.FIREBASE_CLIENT_ID,
    auth_uri: process.env.FIREBASE_AUTH_URI,
    token_uri: process.env.FIREBASE_TOKEN_URI,
    auth_provider_x509_cert_url: process.env.FIREBASE_AUTH_PROVIDER_X509_CERT_URL,
    client_x509_cert_url: process.env.FIREBASE_CLIENT_X509_CERT_URL,
  };
  admin.initializeApp({
    credential: admin.credential.cert(serviceAccount as any),
  });
}

export async function POST(request: Request) {
  try {
    const { to, title, message, deviceId, requestedValue, requestedBy } = await request.json();
    if (!title || !message) {
      return NextResponse.json({ error: 'Title and message are required' }, { status: 400 });
    }

    if (to === 'superusers') {
      // Send to all superusers
      const superuserRes = await postgresClient.query('SELECT user_id, fcm_token, email FROM users WHERE role_id = 0 AND fcm_token IS NOT NULL');
      const tokens = superuserRes.rows.map(row => row.fcm_token);
      if (tokens.length) {
        const payload = {
          notification: {
            title,
            body: message,
          },
        };
        for (const token of tokens) {
          try {
            await admin.messaging().send({ token, ...payload });
          } catch (err) { /* log or ignore individual errors */ }
        }
      }
      // If this is a max value change request, insert into pending_max_value_requests and notifications
      if (deviceId && requestedValue && requestedBy) {
        await postgresClient.query(
          'INSERT INTO pending_max_value_requests (device_id, requested_value, requested_by) VALUES ($1, $2, $3)',
          [deviceId, requestedValue, requestedBy]
        );
        // Insert in-app notification for each superuser
        for (const row of superuserRes.rows) {
          await postgresClient.query(
            'INSERT INTO notifications (user_id, title, message) VALUES ($1, $2, $3)',
            [row.user_id, title, message]
          );
        }
      }
      return NextResponse.json({ success: true });
    }

    // Get FCM tokens for the target user(s) from user_devices
    let tokens: string[] = [];
    let user_id: number | null = null;
    if (to === 'all') {
      // Query all user device tokens
      const result = await postgresClient.query('SELECT fcm_token FROM user_devices WHERE fcm_token IS NOT NULL');
      tokens = result.rows.map(row => row.fcm_token);
    } else {
      user_id = to;
      // If 'to' is not a number, treat as email and look up user_id
      if (isNaN(Number(to))) {
        const userResult = await postgresClient.query('SELECT user_id FROM users WHERE email = $1', [to]);
        if (userResult.rows.length === 0) {
          return NextResponse.json({ error: 'User not found' }, { status: 404 });
        }
        user_id = userResult.rows[0].user_id;
      }
      // Query tokens for a specific user_id
      const result = await postgresClient.query('SELECT fcm_token FROM user_devices WHERE user_id = $1 AND fcm_token IS NOT NULL', [user_id]);
      tokens = result.rows.map(row => row.fcm_token);
    }

    // Always insert in-app notification if user_id is found
    if (user_id) {
      await postgresClient.query(
        'INSERT INTO notifications (user_id, title, message) VALUES ($1, $2, $3)',
        [user_id, title, message]
      );
    }

    if (!tokens.length) {
      if (user_id) {
        // In-app notification sent, so return success
        return NextResponse.json({ success: true, inApp: true, push: false });
      }
      return NextResponse.json({ error: 'No FCM tokens found for the target user(s)' }, { status: 404 });
    }

    // Send notification via FCM
    const payload = {
      notification: {
        title,
        body: message,
      },
    };

    const results = [];
    for (const token of tokens) {
      try {
        const response = await admin.messaging().send({
          token,
          ...payload,
        });
        results.push({ token, response });
      } catch (err) {
        results.push({ token, error: err instanceof Error ? err.message : err });
      }
    }

    return NextResponse.json({ success: true, push: true, inApp: !!user_id, results });
  } catch (error) {
    console.error('Error sending push notification:', error);
    return NextResponse.json({ error: 'Failed to send push notification', details: error instanceof Error ? error.message : 'Unknown error' }, { status: 500 });
  }
} 