import { NextResponse } from 'next/server';
import { postgresClient } from '../../../../lib/postgresql-client';

// GET endpoint to fetch pending requests
export async function GET() {
  try {
    const result = await postgresClient.query(
      `SELECT id, device_id, requested_value, requested_by, requested_at, status, reviewed_by, reviewed_at, review_comment 
       FROM pending_max_value_requests 
       WHERE status = 'pending' 
       ORDER BY requested_at DESC`
    );
    return NextResponse.json({ requests: result.rows });
  } catch (error) {
    console.error('Failed to fetch pending requests:', error);
    return NextResponse.json({ error: 'Failed to fetch pending requests' }, { status: 500 });
  }
}

// POST endpoint to create or update pending requests
export async function POST(request: Request) {
  try {
    const body = await request.json();
    
    // If this is an approval/rejection request
    if (body.id && body.action) {
      const { id, action, reviewer, comment } = body;
      if (!id || !action || !['approve', 'reject'].includes(action)) {
        return NextResponse.json({ error: 'Invalid input' }, { status: 400 });
      }

      // Update the request status
      await postgresClient.query(
        'UPDATE pending_max_value_requests SET status = $1, reviewed_by = $2, reviewed_at = NOW(), review_comment = $3 WHERE id = $4',
        [action === 'approve' ? 'approved' : 'rejected', reviewer, comment || '', id]
      );

      // If approved, update the device's max value
      if (action === 'approve') {
        const reqRes = await postgresClient.query(
          'SELECT device_id, requested_value FROM pending_max_value_requests WHERE id = $1',
          [id]
        );
        
        if (reqRes.rows.length) {
          const { device_id, requested_value } = reqRes.rows[0];
          
          // Ensure device exists in device_max_value
          await postgresClient.query(
            `INSERT INTO device_max_value (device_name, max_value)
             VALUES ($1, $2)
             ON CONFLICT (device_name) DO NOTHING`,
            [device_id, requested_value]
          );
          
          // Update max_value and reset threshold flags
          await postgresClient.query(
            'UPDATE device_max_value SET max_value = $1, "80%" = false, "85%" = false, "90%" = false, "100%" = false WHERE device_name = $2',
            [requested_value, device_id]
          );
        }
      }
      
      return NextResponse.json({ success: true });
    }
    
    // If this is a new request
    const { deviceId, requestedValue, requestedBy } = body;
    if (!deviceId || !requestedValue || !requestedBy) {
      return NextResponse.json({ error: 'Missing required fields' }, { status: 400 });
    }

    // Create new pending request
    await postgresClient.query(
      `INSERT INTO pending_max_value_requests 
       (device_id, requested_value, requested_by, requested_at, status)
       VALUES ($1, $2, $3, NOW(), 'pending')`,
      [deviceId, requestedValue, requestedBy]
    );

    return NextResponse.json({ success: true });
  } catch (error) {
    console.error('Failed to handle pending request:', error);
    return NextResponse.json({ error: 'Failed to handle pending request' }, { status: 500 });
  }
} 