import { Pool, QueryResult, QueryResultRow } from 'pg';

interface UserData {
  device_id: string;
  role_id: number;
  payment_history_id: string;
}

class PostgreSQLClient {
  private pool: Pool;

  constructor() {
    const host = process.env.POSTGRES_HOST;
    const port = process.env.POSTGRES_PORT ? parseInt(process.env.POSTGRES_PORT, 10) : 5432;
    const database = process.env.POSTGRES_DATABASE;
    const user = process.env.POSTGRES_USER;
    const password = process.env.POSTGRES_PASSWORD;

    if (!host || !database || !user || !password) {
      console.error('PostgreSQL configuration is missing required environment variables (HOST, DATABASE, USER, PASSWORD).');
      // Avoid creating the pool if configuration is missing
      // A more robust solution might throw an error or have a different initialization state.
       this.pool = new Pool(); // Create an empty pool to avoid crashing on access, but queries will fail.
       console.warn("PostgreSQL client initialized with incomplete configuration. Queries will likely fail.");
    } else {
       this.pool = new Pool({
         host,
         port,
         database,
         user,
         password,
         ssl: process.env.POSTGRES_SSL === 'true' ? { rejectUnauthorized: false } : false,
       });

        this.pool.on('error', (err, client) => {
            console.error('Unexpected error on idle PostgreSQL client', err);
            // process.exit(-1); // Consider how to handle critical pool errors
        });
        //console.log("PostgreSQL client pool initialized successfully.");
    }
  }

  async query<T extends QueryResultRow = any>(text: string, params?: any[]): Promise<QueryResult<T>> {
    if (!process.env.POSTGRES_HOST) { // Quick check if pool was likely not configured correctly
        throw new Error("PostgreSQL client is not properly configured.");
    }
    const start = Date.now();
    try {
      const res = await this.pool.query<T>(text, params);
      const duration = Date.now() - start;
      //console.log('Executed PostgreSQL query', { text, duration, rows: res.rowCount });
      return res;
    } catch (error: any) {
      console.error('Error executing PostgreSQL query:', error);
      throw new Error(`Database query failed: ${error.message}`);
    }
  }

  // Example function to get max value for a device
  // Return type can be number or string, as pg might return numeric types as strings
  async getDeviceMaxValue(deviceId: string): Promise<number | string | null> {
    const queryText = `
      SELECT max_value
      FROM device_max_value
      WHERE device_name = $1
      LIMIT 1;
    `;
    try {
      // Adjust the expected type to potentially include string
      const result = await this.query<{ max_value: number | string }>(queryText, [deviceId]);
      if (result.rows.length > 0) {
        // Return the raw value, let the API route handle parsing
        return result.rows[0].max_value;
      }
      return null; // Device not found or no max value set
    } catch (error) {
      console.error(`Error fetching max value for device ${deviceId}:`, error);
      // Depending on requirements, you might want to return null or re-throw
      return null;
    }
  }

  async getUserData(email: string, uid: string): Promise<UserData | null> {
    const queryText = `
      SELECT device_id, role_id, payment_history_id
      FROM users
      WHERE email = $1 AND uid = $2
      LIMIT 1;
    `;
    try {
      const result = await this.query<UserData>(queryText, [email, uid]);
      if (result.rows.length > 0) {
        return result.rows[0];
      }
      return null;
    } catch (error) {
      console.error(`Error fetching user data for email ${email}:`, error);
      return null;
    }
  }

  // Gracefully disconnect the pool
  async disconnect(): Promise<void> {
     if (this.pool) {
       await this.pool.end();
       //console.log("PostgreSQL client pool disconnected.");
     }
  }
}

// Export a singleton instance
export const postgresClient = new PostgreSQLClient();
