import React from 'react';
import { useTranslation } from 'react-i18next';
import { BarChart3, CreditCard, Bell, ShoppingCart, Settings } from 'lucide-react';
import { useAuth } from '../hooks/use-auth';

interface ManagerTab {
  id: string;
  label: string;
  icon?: any;
}

interface ManagerNavigationProps {
  activeTab: string;
  setActiveTab: (tabId: string) => void;
  tabs: ManagerTab[];
  menu: Array<{ id: string; label: string; tabId: string }>;
}

const tabIcons: Record<string, any> = {
  dashboard: BarChart3,
  'payment-history': CreditCard,
  'system-notifications': Bell,
  'pending-orders': ShoppingCart,
  order: ShoppingCart,
  'user-settings': Settings,
  'system-settings': Settings,
};

export function ManagerNavigation({ activeTab, setActiveTab, tabs, menu }: ManagerNavigationProps) {
  const { t } = useTranslation();
  const { user, userRoleId, orderTab } = useAuth();

  if (userRoleId !== 1) return null;

  // Фильтруем табы по orderTab
  const filteredTabs = tabs.filter(tab => tab.id !== 'order' || orderTab !== false);

  const navTabs: ManagerTab[] = filteredTabs.map(tab => ({
    ...tab,
    icon: tabIcons[tab.id] || BarChart3,
  }));

  return (
    <div className="flex flex-col lg:flex-row w-full">
      {/* Desktop Navigation */}

      {/* Mobile Navigation */}
      <div className="lg:hidden fixed bottom-0 left-0 right-0 z-50 bg-background border-t">
        <div className="flex justify-around items-center h-14 sm:h-16 px-2">
          {navTabs.map((tab) => {
            const IconComponent = tab.icon;
            return (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id)}
                className={`flex flex-col items-center justify-center flex-1 h-full transition-colors px-1 ${
                  activeTab === tab.id
                    ? 'text-white bg-[#1B526A]'
                    : 'text-muted-foreground hover:text-primary'
                }`}
              >
                <div className="flex items-center justify-center h-5 w-5 sm:h-6 sm:w-6">
                  <IconComponent color={activeTab === tab.id ? 'white' : undefined} />
                </div>
                <span className="text-xs sm:text-sm mt-1 font-medium text-center">
                  {t(tab.label)}
                </span>
              </button>
            );
          })}
        </div>
      </div>
    </div>
  );
} 